/*******************************************************************************
* Copyright (C) 2025 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) IE Sparse BLAS C
*             example for mkl_sparse_convert_{csc, coo} functions and
*             for mkl_sparse_d_export_{csc, coo} functions
*
********************************************************************************
*
* Example program for using Intel oneMKL Inspector-Executor Sparse BLAS routines
* for conversions between different matrix formats.
*
* The following Inspector Executor Sparse Blas routines are used in the example:
*
*   Initialization/Destruction stage:
*          mkl_sparse_d_create_csr
*          mkl_sparse_destroy
*
*   Format conversion functions:
*          mkl_sparse_convert_csc   mkl_sparse_convert_coo
*
*   Sparse matrix export functions:
*          mkl_sparse_d_export_csc  mkl_sparse_d_export_coo
*
*   Sparse matrix-vector multiplication function:
*          mkl_sparse_d_mv
*
*   Sparse matrix ordering function:
*          mkl_sparse_order
*
* Consider the matrix A below to be represented in multiple sparse formats
* (see 'Sparse Matrix Storage Schemes' in the Intel oneMKL Reference Manual):
*
*       |  1  -2   0   0 |
*       |  3  -4   0   0 |
*   A = |  0   0   5  -6 |.
*       |  0   0   7  -8 |
*       |  9 -10   0   0 |
*       | 11 -12   0   0 |
*
*  A compressed sparse row (CSR) representation of the matrix with
*  three arrays is:
*
*     csrNrows  = 6
*     csrNcols  = 4
*     csrNnz    = 12
*     csrIndex  = SPARSE_INDEX_BASE_ZERO
*     csrRowPtr = (0       2       4       6       8      10      12)
*     csrColIdx = (0   1   0   1   2   3   2   3   0   1   0   1)
*     csrValues = (1  -2   3  -4   5  -6   7  -8   9 -10  11 -12)
*
*  A coordinate format (COO) representation of the matrix is:
*
*     cooNrows  = 6
*     cooNcols  = 4
*     cooNnz    = 12
*     cooIndex  = SPARSE_INDEX_BASE_ZERO
*     cooRowIdx = (0   0   1   1   2   2   3   3   4   4   5   5)
*     cooColIdx = (0   1   0   1   2   3   2   3   0   1   0   1)
*     cooValues = (1  -2   3  -4   5  -6   7  -8   9 -10  11 -12)
*
*  A compressed sparse column (CSC) representation of the matrix with
*  three arrays is:
*
*     cscNrows  = 6
*     cscNcols  = 4
*     cscNnz    = 12
*     cscIndex  = SPARSE_INDEX_BASE_ZERO
*     cscColPtr = (0               4               8      10      12)
*     cscRowIdx = (0   1   4   5   0   1   4   5   2   3   2   3)
*     cscValues = (1   3   9  11  -2  -4 -10 -12   5   7  -6  -8)
*
*  This example presents:
*    * mkl_sparse_convert_{csc/coo}() usage to convert a sparse matrix:
*      * from CSR to CSC,
*      * from CSR to COO,
*    * mkl_sparse_d_export_{csc/coo}() usage to extract CSC and COO arrays,
*
*  Note that we will call mkl_sparse_order() after conversion to CSC so that when printing out
*  we have it printed in sorted order in this example.  This is not strictly necessary as
*  most algorithms work with unsorted data (exception is unoptimized TRSV/M which requires lower and
*  upper parts to be separated within each row/column for CSR/CSC/BSR formats.  Algorithms can
*  sometimes be more efficient with sorted data due to better memory access patterns.
*
********************************************************************************
*/
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include "mkl.h"

#ifdef MKL_ILP64
#define INT_PRINT_FORMAT "%4lld"
#else
#define INT_PRINT_FORMAT "%4d"
#endif

void print_int_value(const char *name, MKL_INT val) {
    printf("\t\t%s = " INT_PRINT_FORMAT "\n", name, val);
}

void print_int_array(const char *name, MKL_INT *array, MKL_INT len) {
    printf("\t\t%s =", name);
    for (MKL_INT i = 0; i < len; ++i) {
        printf(INT_PRINT_FORMAT ",", array[i]);
    }
    printf("\n");
}

void print_index(const char *name, sparse_index_base_t idxBase) {
    printf("\t\t%s = %s\n", name, idxBase == SPARSE_INDEX_BASE_ZERO ? "SPARSE_INDEX_BASE_ZERO" : "SPARSE_INDEX_BASE_ONE");
}

void print_flt_array(const char *name, double *array, MKL_INT len) {
    printf("\t\t%s =", name);
    for (MKL_INT i = 0; i < len; ++i) {
        printf("%4.0f,", array[i]);
    }
    printf("\n");
}

int main() {
    //*******************************************************************************
    //     Declaration and initialization of parameters for sparse representation of
    //     the matrix A in the COO format:
    //*******************************************************************************
#define M 6
#define N 4
#define NNZ 12

    MKL_INT csrNrows = M, csrNcols = N, csrNnz = NNZ;
    MKL_INT cooNrows = 0, cooNcols = 0, cooNnz = 0;
    MKL_INT cscNrows = 0, cscNcols = 0, cscNnz = 0;
    sparse_index_base_t csrIdxBase = SPARSE_INDEX_BASE_ZERO;
    sparse_index_base_t cscIdxBase, cooIdxBase;

    struct matrix_descr descrA;
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;

    MKL_INT csc_passed = 0, coo_passed = 0;
    double TOL = 1.0e-6;
    double csrSum = 0.0, cscSum = 0.0, cooSum = 0.0;

    //*******************************************************************************
    //    Vectors for testing correctness of matrix formats
    //*******************************************************************************
    double x_N[N] = {1, 1, 1, 1};
    double y_M[M] = {0, 0, 0, 0, 0, 0};

    //*******************************************************************************
    //    Arrays for the CSR representation of the matrix A
    //*******************************************************************************

    MKL_INT csrRowPtr[M+1] = { 0, 2, 4, 6, 8, 10, 12 };
    MKL_INT csrColIdx[NNZ] = { 0, 1, 0, 1, 2, 3, 2, 3, 0, 1, 0, 1 };
    double csrValues[NNZ]  = { 1, -2, 3, -4, 5, -6, 7, -8, 9, -10, 11, -12 };

    //*******************************************************************************
    //    Sparse format arrays of the matrix A for COO/CSC formats to be filled later
    //*******************************************************************************

    MKL_INT *cscColStart = NULL, *cscColEnd = NULL, *cscRowIdx = NULL;
    MKL_INT *cooRowIdx = NULL, *cooColIdx = NULL;
    double *cscValues = NULL, *cooValues = NULL;

    // Sparse matrix handles to be used for conversion to different formats.
    // Note that we recommend to always initialize them to NULL before using.
    sparse_matrix_t csrA = NULL, cscA = NULL, cooA = NULL;

    sparse_status_t status;
    int exit_status = 0;

    printf( "\nEXAMPLE PROGRAM for matrix format conversion routines from IE Sparse BLAS\n" );
    printf( "-------------------------------------------------------------------------\n" );

    //******************************************************************************
    //    Create CSR sparse matrix handle
    //******************************************************************************
    status = mkl_sparse_d_create_csr(&csrA, csrIdxBase, csrNrows, csrNcols,
                                     csrRowPtr, csrRowPtr+1, csrColIdx, csrValues);

    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_create_csr: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //******************************************************************************
    //    Setup Test   csrSum = ones_M' * A * ones_N =  sum_i { csrValues[i] }
    //******************************************************************************
    for (MKL_INT i = 0; i < csrNnz; ++i) {
        csrSum += csrValues[i];
    }

    //******************************************************************************
    //    Print input CSR Matrix
    //******************************************************************************
    printf("\t[Input] Matrix arrays in CSR format:\n");
    print_int_value("nrows", csrNrows);
    print_int_value("ncols", csrNcols);
    print_int_value("nnz  ", csrNnz);
    print_index(    "index", csrIdxBase);
    print_int_array("csrRowPtr", csrRowPtr, csrNrows+1);
    print_int_array("csrColIdx", csrColIdx, csrNnz);
    print_flt_array("csrValues", csrValues, csrNnz);

    //******************************************************************************
    //    Convert from CSR to COO format
    //******************************************************************************
    status = mkl_sparse_convert_coo(csrA, SPARSE_OPERATION_NON_TRANSPOSE, &cooA);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_convert_coo: %d \n", status);
        exit_status = 1;
        goto exit;
    }
    status = mkl_sparse_d_export_coo(cooA, &cooIdxBase, &cooNrows, &cooNcols,
                                     &cooNnz, &cooRowIdx, &cooColIdx, &cooValues);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_export_coo: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //******************************************************************************
    //    Start Test of COO matrix
    //******************************************************************************
    status = mkl_sparse_d_mv(SPARSE_OPERATION_NON_TRANSPOSE, 1.0, cooA, descrA, x_N, 0.0, y_M);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }
    for (MKL_INT i = 0; i < M; ++i) {
        cooSum += y_M[i];
    }
    if ( fabs(csrSum - cooSum) < TOL ) {
        coo_passed = 1;
    }

    //******************************************************************************
    //    Print output COO Matrix
    //******************************************************************************
    printf("\t[Output] Matrix arrays in COO format:\n");
    print_int_value("nrows", cooNrows);
    print_int_value("ncols", cooNcols);
    print_int_value("nnz  ", cooNnz);
    print_index(    "index", cooIdxBase);
    print_int_array("cooRowIdx", cooRowIdx, cooNnz);
    print_int_array("cooColIdx", cooColIdx, cooNnz);
    print_flt_array("cooValues", cooValues, cooNnz);
    printf("\n");
    if (coo_passed) printf("\t COO Test Passed\n");
    else printf("\t COO Test Failed: CSR sum = %4.0f, but COO sum = %4.0f \n", csrSum, cooSum);
    printf("\n");

    //******************************************************************************
    //    Convert from CSR to CSC format
    //******************************************************************************
    status = mkl_sparse_convert_csc(csrA, SPARSE_OPERATION_NON_TRANSPOSE, &cscA);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_convert_csc: %d \n", status);
        exit_status = 1;
        goto exit;
    }
    status = mkl_sparse_order(cscA);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_order: %d \n", status);
        exit_status = 1;
        goto exit;
    }
    status = mkl_sparse_d_export_csc(cscA, &cscIdxBase, &cscNrows, &cscNcols,
                                     &cscColStart, &cscColEnd, &cscRowIdx, &cscValues);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_export_coo: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //******************************************************************************
    //    Start Test of CSC matrix
    //******************************************************************************
    status = mkl_sparse_d_mv(SPARSE_OPERATION_NON_TRANSPOSE, 1.0, cscA, descrA, x_N, 0.0, y_M);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }
    for (MKL_INT i = 0; i < M; ++i) {
        cscSum += y_M[i];
    }
    if ( fabs(csrSum - cscSum) < TOL ) {
        csc_passed = 1;
    }

    //******************************************************************************
    //    Print output CSC Matrix
    //******************************************************************************
    cscNnz = cscColEnd[cscNcols-1] - cscColStart[0];
    printf("\t[Output] Matrix arrays CSC format:\n");
    print_int_value("nrows", cscNrows);
    print_int_value("ncols", cscNcols);
    print_int_value("nnz  ", cscNnz);
    print_index(    "index", cscIdxBase);
    print_int_array("cscColPtr", cscColStart, cscNcols+1);
    print_int_array("cscRowIdx", cscRowIdx, cscNnz);
    print_flt_array("cscValues", cscValues, cscNnz);
    if (csc_passed) printf("\t CSC Test Passed\n");
    else printf("\t CSC Test Failed: CSR sum = %4.0f, but CSC sum = %4.0f \n", csrSum, cscSum);
    printf("\n");

exit:
    // Release matrix handle and deallocate matrix
    if (csrA) mkl_sparse_destroy(csrA);
    if (cooA) mkl_sparse_destroy(cooA);
    if (cscA) mkl_sparse_destroy(cscA);

    return exit_status;
}
