/*******************************************************************************
* Copyright (C) 2025 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) PARDISO C example
*
********************************************************************************
*/
/* -------------------------------------------------------------------- */
/* Example program to show the use of the "PARDISO" routine with
 * sparse matrices that are read from a Matrix Market file
*/
/* -------------------------------------------------------------------- */
/* */
/* (C) Olaf Schenk, Department of Computer Science, */
/* University of Basel, Switzerland. */
/* Email: olaf.schenk@unibas.ch */
/* -------------------------------------------------------------------- */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "mkl_pardiso.h"
#include "mkl_types.h"
#include "matrix_reader.h"

// Define the format to printf MKL_INT values
#if !defined(MKL_ILP64)
#define IFORMAT "%i"
#else
#define IFORMAT "%lli"
#endif

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)>(b))?(a):(b))

int main (int argc, char **argv)
{

    MKL_INT *A_ia = NULL, *A_ja = NULL;
    double *A_val = NULL;
    MKL_INT nrows, ncols;
    MKL_INT mtype;
    MKL_INT is_symm;
    sparse_status_t stat;

    char *matrix_filename = argv[1];
    if(argc <= 1)
    {
        printf("\nYou must specify a Matrix Market file as 1-st parameter.\n");
        printf("\nUsage: %s <Matrix Market file>\n", argv[0]);
        return 1;
    }
    else
    {
        //get filename provided by user
        matrix_filename = argv[1];
    }


    // Descriptor of main sparse matrix properties
    struct matrix_descr descrA;
    sparse_matrix_t A_csr_read = NULL;
    sparse_matrix_t A_csr = NULL;
    MKL_INT indexing = 1;
    sparse_operation_t operation = SPARSE_OPERATION_NON_TRANSPOSE;
    A_csr_read = d_matrix_reader(matrix_filename, indexing, operation, &is_symm);
    if(A_csr_read == NULL)
    {
        printf("Error in matrix reader.\n");
        exit(1);
    }
    descrA.diag = SPARSE_DIAG_NON_UNIT;
    //MatrixMarket file returns lower triangular matrix for symmetric matrices.
    //Use transpose to convert to upper triangular matrix for PARDISO.
    if(is_symm)
    {
        mtype = -2; //PARDISO mtype
        operation = SPARSE_OPERATION_TRANSPOSE;
        stat = mkl_sparse_convert_csr ( A_csr_read, operation, &A_csr );
        if(stat != SPARSE_STATUS_SUCCESS)
        {
            printf("Error in mkl_sparse_convert_csr, while transposing.\n"); fflush(0);
            exit(1);
        }
        descrA.type = SPARSE_MATRIX_TYPE_SYMMETRIC;
        descrA.mode = SPARSE_FILL_MODE_UPPER;
    }
    else
    {
        mtype = 11; //PARDISO mtype
        A_csr = A_csr_read;
        descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
        descrA.mode = SPARSE_FILL_MODE_UPPER;
    }
    sparse_index_base_t A_indexing;
    //sort non-zeros within row accoring to column indices
    stat = mkl_sparse_order(A_csr);
    if(stat != SPARSE_STATUS_SUCCESS)
    {
        printf("Error in mkl_sparse_order.\n"); fflush(0);
        exit(1);
    }
    MKL_INT *ia2 = NULL;
    stat = mkl_sparse_d_export_csr (A_csr, &A_indexing, &nrows, &ncols, &A_ia, &ia2, &A_ja, &A_val);
    if(stat != SPARSE_STATUS_SUCCESS || (A_indexing == SPARSE_INDEX_BASE_ZERO))
    {
        printf("Error in mkl_sparse_d_export_csr\n"); fflush(0);
        exit(1);
    }

    int ALIGNMENT = 64;
    /* RHS and solution vectors. */
    double *b, *x, *res;
    b = (double*) mkl_malloc(nrows*sizeof(double), ALIGNMENT);
    x = (double*) mkl_malloc(nrows*sizeof(double), ALIGNMENT);
    res = (double*) mkl_malloc(nrows*sizeof(double), ALIGNMENT);
    MKL_INT nrhs = 1;     /* Number of right hand sides. */
    /* Internal solver memory pointer pt, */
    void *pt[64];
    /* Pardiso control parameters. */
    MKL_INT iparm[64];
    MKL_INT maxfct, mnum, phase, error, msglvl;
    /* Auxiliary variables. */
    MKL_INT i;
    double ddum;          /* Double dummy */
    MKL_INT idum;         /* Integer dummy. */
/* -------------------------------------------------------------------- */
/* .. Setup Pardiso control parameters. */
/* -------------------------------------------------------------------- */
    for ( i = 0; i < 64; i++ )
    {
        iparm[i] = 0;
    }
    iparm[0] = 1;         /* No solver default */
    iparm[1] = 3;         /* Fill-in reordering from METIS */
    iparm[3] = 0;         /* No iterative-direct algorithm */
    iparm[4] = 0;         /* No user fill-in reducing permutation */
    iparm[5] = 0;         /* Write solution into x */
    iparm[6] = 0;         /* Not in use */
    iparm[7] = 2;         /* Max numbers of iterative refinement steps */
    iparm[8] = 0;         /* Not in use */
    iparm[9] = 13;        /* Perturb the pivot elements with 1E-13 */
    iparm[10] = 1;        /* Use nonsymmetric permutation and scaling MPS */
    iparm[11] = 0;        /* Not in use */
    iparm[12] = 0;        /* Maximum weighted matching algorithm is switched-off (default for symmetric). Try iparm[12] = 1 in case of inappropriate accuracy */
    iparm[13] = 0;        /* Output: Number of perturbed pivots */
    iparm[14] = 0;        /* Not in use */
    iparm[15] = 0;        /* Not in use */
    iparm[16] = 0;        /* Not in use */
    iparm[17] = -1;       /* Output: Number of nonzeros in the factor LU */
    iparm[18] = -1;       /* Output: Mflops for LU factorization */
    iparm[19] = 0;        /* Output: Numbers of CG Iterations */
    maxfct = 1;           /* Maximum number of numerical factorizations. */
    mnum = 1;             /* matrix number */
    msglvl = 0;           /* Print statistical information in file */
    error = 0;            /* Initialize error flag */
/* -------------------------------------------------------------------- */
/* .. Initialize the internal solver memory pointer. This is only */
/* necessary for the FIRST call of the PARDISO solver. */
/* -------------------------------------------------------------------- */
    for ( i = 0; i < 64; i++ )
    {
        pt[i] = 0;
    }
/* -------------------------------------------------------------------- */
/* .. Reordering and Symbolic Factorization. This step also allocates */
/* all memory that is necessary for the factorization. */
/* -------------------------------------------------------------------- */
    phase = 11;
    PARDISO (pt, &maxfct, &mnum, &mtype, &phase,
             &nrows, A_val, A_ia, A_ja, &idum, &nrhs, iparm, &msglvl, &ddum, &ddum, &error);
    if ( error != 0 )
    {
        printf ("\nERROR during symbolic factorization: " IFORMAT, error);
        exit (1);
    }
    printf ("\nReordering completed ... ");
    printf ("\nNumber of nonzeros in factors = " IFORMAT, iparm[17]);
    printf ("\nNumber of factorization MFLOPS = " IFORMAT, iparm[18]);
/* -------------------------------------------------------------------- */
/* .. Numerical factorization. */
/* -------------------------------------------------------------------- */
    phase = 22;
    PARDISO (pt, &maxfct, &mnum, &mtype, &phase,
             &nrows, A_val, A_ia, A_ja, &idum, &nrhs, iparm, &msglvl, &ddum, &ddum, &error);
    if ( error != 0 )
    {
        printf ("\nERROR during numerical factorization: " IFORMAT, error);
        exit (2);
    }
    printf ("\nFactorization completed ... ");
/* -------------------------------------------------------------------- */
/* .. Back substitution and iterative refinement. */
/* -------------------------------------------------------------------- */
    phase = 33;
    iparm[7] = 2;         /* Max numbers of iterative refinement steps. */
    /* Set right hand side to one. */
    for ( i = 0; i < nrows; i++ )
    {
        b[i] = 1;
    }
    PARDISO (pt, &maxfct, &mnum, &mtype, &phase,
             &nrows, A_val, A_ia, A_ja, &idum, &nrhs, iparm, &msglvl, b, x, &error);
    if ( error != 0 )
    {
        printf ("\nERROR during solution: " IFORMAT, error);
        exit (3);
    }
    printf ("\nSolve completed ... ");
    printf ("\nThe solution of the system is: ");
    for ( i = 0; i < MIN(nrows, 10); i++ )
    {
        printf ("\n x [" IFORMAT "] = % f", i, x[i]);
    }
    if(nrows > 10)
    {
        printf ("\n ...\n");
        printf("Results truncated to first 10 rows\n");
    }
    printf ("\n");

    sparse_operation_t transA = SPARSE_OPERATION_NON_TRANSPOSE;
    // Compute residual
    mkl_sparse_d_mv( transA, 1.0, A_csr, descrA, x, 0.0, res);
    double res_norm_sq = 0.0;
    double b_norm_sq = 0.0;
    for (MKL_INT j = 1; j <= nrows; j++ )
    {
        res_norm_sq += (res[j - 1] - b[j - 1]) * (res[j - 1] - b[j - 1]);
        b_norm_sq += b[j - 1] * b[j - 1];
    }
    double rel_res = sqrt (res_norm_sq) / sqrt (b_norm_sq);
    printf ("\nRelative residual = %e\n", rel_res);

/* -------------------------------------------------------------------- */
/* .. Termination and release of memory. */
/* -------------------------------------------------------------------- */
    phase = -1;           /* Release internal memory. */
    PARDISO (pt, &maxfct, &mnum, &mtype, &phase,
             &nrows, A_val, A_ia, A_ja, &idum, &nrhs,
             iparm, &msglvl, &ddum, &ddum, &error);

    if(A_csr)
    {
        //destroy A_csr, A_ia, A_ja, and A_val
        stat = mkl_sparse_destroy(A_csr);
        if(stat != SPARSE_STATUS_SUCCESS)
        {
            printf("Error in mkl_sparse_detroy.\n"); fflush(0);
            exit(1);
        }
    }
    //destroy non-transposed version
    if(is_symm)
    {
        stat = mkl_sparse_destroy(A_csr_read);
        if(stat != SPARSE_STATUS_SUCCESS)
        {
            printf("Error in mkl_sparse_detroy.\n"); fflush(0);
            exit(1);
        }
    }

    if(b)
    {
        mkl_free(b);
    }
    if(x)
    {
        mkl_free(x);
    }
    if(res)
    {
        mkl_free(res);
    }
    return 0;
}
