/*******************************************************************************
* Copyright (C) 2021 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP USM API oneapi::mkl::sparse::gemm to perform general sparse matrix-matrix
*       multiplication on a SYCL device (CPU, GPU). This example uses
*       column-major layout for dense matrices X and Y, and CSR format for
*       sparse matrix A.
*
*       Y = alpha * op(A) * op(X) + beta * Y
*
*       where op() is defined by one of
*           oneapi::mkl::transpose::{nontrans,trans,conjtrans}
*
*       The supported floating point data types for gemm are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported transpose operation op() for sparse matrix A are:
*           oneapi::mkl::transpose::nontrans
*           oneapi::mkl::transpose::trans
*           oneapi::mkl::transpose::conjtrans
*
*       The supported transpose operation op() for dense matrix X are:
*           oneapi::mkl::transpose::nontrans
*
*       The supported matrix formats for gemm are:
*           CSR
*           COO (currently only on CPU device)
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"
#include "./include/reference_impls.hpp"

//
// Main example for Sparse Matrix-Dense Matrix Multiply consisting of
// initialization of sparse A matrix, and dense X and Y matrices as well as
// scalars alpha and beta.  Then the product
//
// Y = alpha * op(A) * op(X) + beta * Y
//
// is performed and finally the results are post processed.
//
template <typename dataType, typename intType>
int run_sparse_blas_example(sycl::queue &q)
{
    bool good = true;

    //
    // handle for sparse matrix
    //
    oneapi::mkl::sparse::matrix_handle_t csrA = nullptr;

    // vectors for keeping track of memory cleanup
    std::vector<intType *> int_ptr_vec;
    std::vector<dataType *> data_ptr_vec;

    try {

        // Initialize data for Sparse Matrix-Dense Matrix Multiply
        oneapi::mkl::transpose opA = oneapi::mkl::transpose::nontrans;
        if constexpr (is_complex<dataType>()) { opA = oneapi::mkl::transpose::conjtrans; }

        oneapi::mkl::transpose opX              = oneapi::mkl::transpose::nontrans;
        oneapi::mkl::layout dense_matrix_layout = oneapi::mkl::layout::col_major;
        oneapi::mkl::index_base index_base_val  = oneapi::mkl::index_base::zero;

        intType int_index = (index_base_val == oneapi::mkl::index_base::zero ? 0 : 1);

        // Matrix data size
        intType nrows_a      = 64;
        intType ncols_a      = 128;
        std::int64_t columns = 256;
        std::int64_t ldx     = (opA == oneapi::mkl::transpose::nontrans) ? ncols_a : nrows_a; // column major
        std::int64_t ldy     = (opA == oneapi::mkl::transpose::nontrans) ? nrows_a : ncols_a; // column major

        double density_val = 0.05;

        // Input matrix in CSR format
        std::vector<intType, mkl_allocator<intType, 64>> ia_vec;
        std::vector<intType, mkl_allocator<intType, 64>> ja_vec;
        std::vector<dataType, mkl_allocator<dataType, 64>> a_vec;

        generate_random_sparse_matrix<dataType, intType>(nrows_a, ncols_a, density_val, ia_vec, ja_vec,
                                                   a_vec, int_index);
        intType nnz_a = ia_vec[nrows_a] - int_index;

        intType *ia, *ja;
        dataType *a, *dnsX, *dnsY, *dnsY_ref;
        intType sizea   = a_vec.size();
        intType sizeja  = ja_vec.size();
        intType sizeia  = ia_vec.size();

        intType nrows_opa = (oneapi::mkl::transpose::nontrans == opA) ? nrows_a : ncols_a;
        intType ncols_opa = (oneapi::mkl::transpose::nontrans == opA) ? ncols_a : nrows_a;

        intType nrows_x = ncols_opa;
        intType ncols_x = columns;
        intType nrows_y = nrows_opa;
        intType ncols_y = columns;

        ia       = sycl::malloc_shared<intType>(sizeia, q);
        ja       = sycl::malloc_shared<intType>(sizeja, q);
        a        = sycl::malloc_shared<dataType>(sizea, q);
        dnsX     = sycl::malloc_shared<dataType>(ldx * ncols_x, q);
        dnsY     = sycl::malloc_shared<dataType>(ldy * ncols_y, q);
        dnsY_ref = sycl::malloc_shared<dataType>(ldy * ncols_y, q);

        if (!ia || !ja || !a || !dnsX || !dnsY || !dnsY_ref) {
            std::string errorMessage =
                "Failed to allocate USM shared memory arrays \n"
                " for CSR A matrix: ia(" + std::to_string((sizeia)*sizeof(intType)) + " bytes)\n"
                "                   ja(" + std::to_string((sizeja)*sizeof(intType)) + " bytes)\n"
                "                   a(" + std::to_string((sizea)*sizeof(dataType)) + " bytes)\n"
                " and dns matrices  dnsX(" + std::to_string((ldx * ncols_x)*sizeof(dataType)) + " bytes)\n"
                "                   dnsY(" + std::to_string((ldy * ncols_y)*sizeof(dataType)) + " bytes)\n"
                "                   dnsY_ref(" + std::to_string((ldy * ncols_y)*sizeof(dataType)) + " bytes)";

            throw std::runtime_error(errorMessage);
        }

        int_ptr_vec.push_back(ia);
        int_ptr_vec.push_back(ja);
        data_ptr_vec.push_back(a);
        data_ptr_vec.push_back(dnsX);
        data_ptr_vec.push_back(dnsY);
        data_ptr_vec.push_back(dnsY_ref);

        rand_matrix<dataType>(dnsX, dense_matrix_layout, nrows_x, ncols_x, ldx);

        for (int i = 0; i < ia_vec.size(); i++)
            ia[i]  = ia_vec[i];

        for (int i = 0; i < ja_vec.size(); i++) {
            ja[i] = ja_vec[i];
            a[i]  = a_vec[i];
        }

        // Init matrices Y and Y_ref
        for (int i = 0; i < ldy * ncols_y; i++) {
            dnsY[i] = set_fp_value(dataType(0.0), dataType(0.0));
            dnsY_ref[i] = set_fp_value(dataType(0.0), dataType(0.0));
        }

        // Set scalar dataType values
        dataType alpha = set_fp_value(dataType(1.0), dataType(0.0));
        dataType beta  = set_fp_value(dataType(0.0), dataType(0.0));

        //
        // Execute Matrix Multiply
        //

        std::cout << "\n\t\tsparse::gemm parameters:\n";
        std::cout << "\t\t\tdense_matrix_layout = " << dense_matrix_layout << std::endl;
        std::cout << "\t\t\topA                 = " << opA << std::endl;
        std::cout << "\t\t\topX                 = " << opX << std::endl;
        std::cout << "\t\t\tnrows               = " << nrows_a << std::endl;
        std::cout << "\t\t\tncols               = " << ncols_a << std::endl;
        std::cout << "\t\t\tnnz                 = " << nnz_a << std::endl;
        std::cout << "\t\t\tcolumns             = " << columns << std::endl;
        std::cout << "\t\t\tldx                 = " << ldx << std::endl;
        std::cout << "\t\t\tldy                 = " << ldy << std::endl;
        std::cout << "\t\t\talpha               = " << alpha << std::endl;
        std::cout << "\t\t\tbeta                = " << beta << std::endl;

        oneapi::mkl::sparse::init_matrix_handle(&csrA);

        auto ev_set = oneapi::mkl::sparse::set_csr_data(q, csrA, nrows_a, ncols_a,
                index_base_val, ia, ja, a);

        auto ev_gemm =
                oneapi::mkl::sparse::gemm(q, dense_matrix_layout, opA, opX,
                                          alpha, csrA, dnsX, columns, ldx, beta, dnsY, ldy, {ev_set});

        auto ev_rel = oneapi::mkl::sparse::release_matrix_handle(q, &csrA, {ev_gemm});

        q.wait();

        //
        // Post Processing
        //

        // Compute reference gemm result.
        // NOTE: Now we support only opX == nontrans case, so we don't pass it as argument.
        prepare_reference_gemm_data(ia, ja, a, nrows_a, ncols_a, nnz_a, int_index,
                opA, alpha, beta, dense_matrix_layout, dnsX, ncols_x, ldx, dnsY_ref, ldy);

        dataType diff  = set_fp_value(dataType(0.0), dataType(0.0));
        dataType diff2 = set_fp_value(dataType(0.0), dataType(0.0));
        dataType *res = dnsY;
        for (intType j = 0; j < ncols_y; j++) {
            intType flops_for_val = 2 * (ceil_div(nnz_a, nrows_y) + 2);
            for (intType i = 0; i < nrows_y; i++) {
                if (opA == oneapi::mkl::transpose::nontrans) {
                    flops_for_val = 2*(ia[i+1] - ia[i] + 2);
                }

                intType index = i + j * ldy;
                if (!check_result(res[index], dnsY_ref[index], flops_for_val, index)) {
                    good = false;
                }
                diff += (dnsY_ref[index] - res[index]) * (dnsY_ref[index] - res[index]);
                diff2 += dnsY_ref[index] * dnsY_ref[index];
            }
        }

        std::cout << "\n\t\t sparse::gemm residual:\n"
                  << "\t\t\t" << diff / diff2 << "\n\tFinished" << std::endl;

        q.wait_and_throw();

    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;
        good = false;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;
        good = false;
    }

    q.wait();

    // backup cleaning of matrix handle and others for if exceptions happened
    oneapi::mkl::sparse::release_matrix_handle(q, &csrA, {}).wait();

    cleanup_arrays<dataType, intType>(data_ptr_vec, int_ptr_vec, q);

    return good ? 0 : 1;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# Sparse Matrix-Dense Matrix Multiply Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Y = alpha * op(A) * op(X) + beta * Y" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A is a sparse matrix in CSR format, X and Y are "
                 "dense matrices"
              << std::endl;
    std::cout << "# and alpha, beta are floating point type precision scalars." << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::gemm" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type,
//  run_sparse_blas_example is run with all supported data types,
//  if any fail, we move on to the next device.
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {
        try {
            sycl::device my_dev;
            bool my_dev_is_found = false;
            get_sycl_device(my_dev, my_dev_is_found, *it);

            if (my_dev_is_found) {
                std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

                // Catch asynchronous exceptions
                auto exception_handler = [](sycl::exception_list exceptions) {
                    for (std::exception_ptr const &e : exceptions) {
                        try {
                            std::rethrow_exception(e);
                        }
                        catch (sycl::exception const &e) {
                            std::cout << "Caught asynchronous SYCL exception: \n"
                                << e.what() << std::endl;
                        }
                    }
                };

                sycl::queue q(my_dev, exception_handler);

                std::cout << "\tRunning with single precision real data type:" << std::endl;
                status |= run_sparse_blas_example<float, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision real data type:" << std::endl;
                    status |= run_sparse_blas_example<double, std::int32_t>(q);
                }

                std::cout << "\tRunning with single precision complex data type:" << std::endl;
                status |= run_sparse_blas_example<std::complex<float>, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision complex data type:" << std::endl;
                    status |= run_sparse_blas_example<std::complex<double>, std::int32_t>(q);
                }

            }
            else {
#ifdef FAIL_ON_MISSING_DEVICES
                std::cout << "No " << sycl_device_names[*it]
                    << " devices found; Fail on missing devices "
                    "is enabled.\n";
                return 1;
#else
                std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                    << sycl_device_names[*it] << " tests.\n";
#endif
            }
        }
        catch (sycl::exception const &e) {
            std::cout << "\t\tCaught SYCL exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }
        catch (std::exception const &e) {
            std::cout << "\t\tCaught std exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }


    } // for loop over devices

    mkl_free_buffers();
    return status;
}
