"use strict";

var _signatureV = require("@aws-sdk/signature-v4");
var _protocolHttp = require("@aws-sdk/protocol-http");
var _sha256Js = require("@aws-crypto/sha256-js");
var _abortController = require("@aws-sdk/abort-controller");
var _credentialProviderNode = require("@aws-sdk/credential-provider-node");
var _nodeHttpHandler = require("@aws-sdk/node-http-handler");
var _queryString = _interopRequireDefault(require("query-string"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const HttpConnector = require('elasticsearch/src/lib/connectors/http');
class HttpAmazonESConnector extends HttpConnector {
  constructor(host, config) {
    super(host, config);
    const protocol = host.protocol;
    const port = host.port;
    this.endpoint = {
      protocol: protocol ? protocol.replace(/:?$/, ':') : 'https:',
      hostname: host.host,
      port: port || (protocol === 'https:' ? 443 : 80),
      path: '/'
    };
    this.awsConfig = config.awsConfig || {};
    this.httpOptions = config.httpOptions || {};
    this.httpClient = new _nodeHttpHandler.NodeHttpHandler(this.httpOptions);
    this.service = config.service || 'es';
  }
  async request(params, cb) {
    const reqParams = this.makeReqParams(params);
    let cancelled = false;
    const controller = new _abortController.AbortController();
    const cancel = () => {
      cancelled = true;
      controller.abort();
    };
    try {
      const creds = await this.getAWSCredentials(reqParams);
      if (cancelled) return;
      let request = this.createRequest(params, reqParams);
      request = await this.signRequest(request, creds);
      const {
        response
      } = await this.httpClient.handle(request, {
        abortSignal: controller.signal
      });
      const body = await this.streamToString(response.body);
      this.log.trace(params.method, reqParams, params.body, body, response.statusCode);
      cb(null, body, response.statusCode, response.headers);
    } catch (err) {
      cb(err);
    }
    return cancel;
  }

  // Helper method to convert readable stream to string
  streamToString(stream) {
    return new Promise((resolve, reject) => {
      const chunks = [];
      stream.on('data', chunk => chunks.push(chunk));
      stream.on('error', reject);
      stream.on('end', () => resolve(Buffer.concat(chunks).toString('utf8')));
    });
  }
  getAWSCredentials(reqParams) {
    var _reqParams$headers;
    if ((_reqParams$headers = reqParams.headers) !== null && _reqParams$headers !== void 0 && _reqParams$headers.auth) {
      const awssigv4Cred = reqParams.headers.auth;
      const {
        accessKeyId,
        secretAccessKey,
        sessionToken
      } = awssigv4Cred.credentials;
      this.service = awssigv4Cred.service;
      delete reqParams.headers.auth;
      return {
        accessKeyId,
        secretAccessKey,
        sessionToken
      };
    }

    // Use default credential provider chain
    return (0, _credentialProviderNode.defaultProvider)()();
  }
  createRequest(params, reqParams) {
    const [pathname = '/', queryStr = ''] = (reqParams.path || '').split('?', 2);
    const queryParams = queryStr ? _queryString.default.parse(queryStr) : undefined;
    const request = new _protocolHttp.HttpRequest({
      ...this.endpoint,
      method: reqParams.method,
      headers: reqParams.headers || {},
      hostname: this.endpoint.hostname,
      query: queryParams
    });
    Object.assign(request, {
      ...reqParams,
      path: pathname.replaceAll(/[!'()*]/g, c => `%${c.charCodeAt(0).toString(16).toUpperCase()}`)
    });
    const body = params.body;
    if (body) {
      const contentLength = Buffer.isBuffer(body) ? body.length : Buffer.byteLength(body);
      request.headers['Content-Length'] = contentLength.toString();
      request.body = body;
    }
    request.headers.Host = this.endpoint.hostname;
    return request;
  }
  async signRequest(request, credentials) {
    const signer = new _signatureV.SignatureV4({
      credentials,
      region: this.awsConfig.region,
      service: this.service,
      sha256: _sha256Js.Sha256
    });
    return signer.sign(request);
  }
}
module.exports = HttpAmazonESConnector;