"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerADRoutes = registerADRoutes;
var _lodash = require("lodash");
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _adHelpers = require("./utils/adHelpers");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                                                                                                                          * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                                                                                                                          * compatible open source license.
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                                                                                                                          * GitHub history for details.
                                                                                                                                                                                                                                                                                                                          */
function registerADRoutes(apiRouter, adService) {
  // create detector
  apiRouter.post('/detectors', adService.putDetector);
  apiRouter.post('/detectors/{dataSourceId}', adService.putDetector);

  // put detector
  apiRouter.put('/detectors/{detectorId}', adService.putDetector);
  apiRouter.put('/detectors/{detectorId}/{dataSourceId}', adService.putDetector);
  apiRouter.post('/detectors/_search', adService.searchDetector);

  // post search anomaly results
  apiRouter.post('/detectors/results/_search', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{dataSourceId}', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}/{dataSourceId}', adService.searchResults);

  // list detectors
  apiRouter.get('/detectors/_list', adService.getDetectors);
  apiRouter.get('/detectors/_list/{dataSourceId}', adService.getDetectors);

  // preview detector
  apiRouter.post('/detectors/preview', adService.previewDetector);
  apiRouter.post('/detectors/preview/{dataSourceId}', adService.previewDetector);

  // get detector anomaly results
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}/{dataSourceId}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{dataSourceId}', adService.getAnomalyResults);

  // delete detector
  apiRouter.delete('/detectors/{detectorId}', adService.deleteDetector);
  apiRouter.delete('/detectors/{detectorId}/{dataSourceId}', adService.deleteDetector);

  // start detector
  apiRouter.post('/detectors/{detectorId}/start', adService.startDetector);
  apiRouter.post('/detectors/{detectorId}/start/{dataSourceId}', adService.startDetector);

  // stop detector
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}', adService.stopDetector);
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}/{dataSourceId}', adService.stopDetector);
  apiRouter.get('/detectors/{detectorId}/_profile', adService.getDetectorProfile);

  // get detector
  apiRouter.get('/detectors/{detectorId}', adService.getDetector);
  apiRouter.get('/detectors/{detectorId}/{dataSourceId}', adService.getDetector);

  // match detector
  apiRouter.get('/detectors/{detectorName}/_match', adService.matchDetector);
  apiRouter.get('/detectors/{detectorName}/_match/{dataSourceId}', adService.matchDetector);

  // get detector count
  apiRouter.get('/detectors/_count', adService.getDetectorCount);
  apiRouter.get('/detectors/_count/{dataSourceId}', adService.getDetectorCount);

  // post get top anomaly results
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}', adService.getTopAnomalyResults);
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}/{dataSourceId}', adService.getTopAnomalyResults);

  // validate detector
  apiRouter.post('/detectors/_validate/{validationType}', adService.validateDetector);
  apiRouter.post('/detectors/_validate/{validationType}/{dataSourceId}', adService.validateDetector);
}
class AdService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "deleteDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.deleteDetector', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - deleteDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "previewDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.previewDetector', {
          body: requestBody
        });
        const transformedKeys = (0, _helpers.mapKeysDeep)(response, _helpers.toCamel);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            //@ts-ignore
            response: (0, _adHelpers.anomalyResultMapper)(transformedKeys.anomalyResult)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - previewDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "putDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;

        //@ts-ignore
        const ifSeqNo = request.body.seqNo;
        //@ts-ignore
        const ifPrimaryTerm = request.body.primaryTerm;
        const requestBody = JSON.stringify((0, _adHelpers.convertDetectorKeysToSnakeCase)(request.body));
        let params = {
          detectorId: detectorId,
          ifSeqNo: ifSeqNo,
          ifPrimaryTerm: ifPrimaryTerm,
          body: requestBody
        };
        let response;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        if ((0, _lodash.isNumber)(ifSeqNo) && (0, _lodash.isNumber)(ifPrimaryTerm)) {
          response = await callWithRequest('ad.updateDetector', params);
        } else {
          response = await callWithRequest('ad.createDetector', {
            body: params.body
          });
        }
        const resp = {
          ...response.anomaly_detector,
          id: response._id,
          primaryTerm: response._primary_term,
          seqNo: response._seq_no
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: (0, _adHelpers.convertDetectorKeysToCamelCase)(resp)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - PutDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "validateDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          validationType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const response = await callWithRequest('ad.validateDetector', {
          body: requestBody,
          validationType: validationType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - validateDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const detectorResponse = await callWithRequest('ad.getDetector', {
          detectorId
        });

        // Populating static detector fields
        const staticFields = {
          id: detectorResponse._id,
          primaryTerm: detectorResponse._primary_term,
          seqNo: detectorResponse._seq_no,
          ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse.anomaly_detector)
        };

        // Get real-time and historical task info to populate the
        // task and job-related fields
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};
        try {
          const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
          realtimeTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, true)
          });
          historicalTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTask = (0, _lodash.get)((0, _lodash.get)(realtimeTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const historicalTask = (0, _lodash.get)((0, _lodash.get)(historicalTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const taskAndJobFields = (0, _adHelpers.convertTaskAndJobFieldsToCamelCase)(realtimeTask, historicalTask, detectorResponse.anomaly_detector_job);

        // Combine the static and task-and-job-related fields into
        // a final response
        const finalResponse = {
          ...staticFields,
          ...taskAndJobFields
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: finalResponse
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get detector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "startDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        var _request$body, _request$body2;
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        //@ts-ignore
        const startTime = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.startTime;
        //@ts-ignore
        const endTime = (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.endTime;
        let requestParams = {
          detectorId: detectorId
        };
        let requestPath = 'ad.startDetector';
        // If a start and end time are passed: we want to start a historical analysis
        if ((0, _lodash.isNumber)(startTime) && (0, _lodash.isNumber)(endTime)) {
          requestParams = {
            ...requestParams,
            body: {
              start_time: startTime,
              end_time: endTime
            }
          };
          requestPath = 'ad.startHistoricalDetector';
        }
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, requestParams);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - startDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "stopDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.stopHistoricalDetector' : 'ad.stopDetector';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - stopDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectorProfile", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.detectorProfile', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - detectorProfile', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify(request.body);
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectors = (0, _lodash.get)(response, 'hits.hits', []).map(detector => ({
          ...(0, _adHelpers.convertDetectorKeysToCamelCase)(detector._source),
          id: detector._id,
          seqNo: detector._seq_no,
          primaryTerm: detector._primary_term
        }));
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectors
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        var {
          resultIndex,
          onlyQueryCustomResultIndex
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
          // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
          resultIndex = '';
        }
        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const requestBody = JSON.stringify(request.body);
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('ad.searchResults', {
          body: requestBody
        }) : await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search anomaly result', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectors", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          from = 0,
          size = 20,
          search = '',
          indices = '',
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = 'name',
          dataSourceId = ''
        } = request.query;
        const mustQueries = [];
        if (search.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['name', 'description'],
              default_operator: 'AND',
              query: `*${search.trim().split('-').join('* *')}*`
            }
          });
        }
        if (indices.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['indices'],
              default_operator: 'AND',
              query: `*${indices.trim().split('-').join('* *')}*`
            }
          });
        }
        //Allowed sorting columns
        const sortQueryMap = {
          name: {
            'name.keyword': sortDirection
          },
          indices: {
            'indices.keyword': sortDirection
          },
          lastUpdateTime: {
            last_update_time: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }
        //Preparing search request
        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              must: mustQueries
            }
          }
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);

        //Get all detectors from search detector API
        const allDetectors = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, detectorResponse) => ({
          ...acc,
          [detectorResponse._id]: {
            id: detectorResponse._id,
            primaryTerm: detectorResponse._primary_term,
            seqNo: detectorResponse._seq_no,
            ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse._source)
          }
        }), {});
        const allDetectorsMap = Object.values(allDetectors).reduce((acc, detector) => ({
          ...acc,
          [detector.id]: detector
        }), {});

        //Given each detector from previous result, get aggregation to power list
        const allDetectorIds = Object.keys(allDetectorsMap);
        let requestParams = {
          // If specifying result index, will query anomaly result from both default and custom result indices.
          // If no valid result index specified, just query anomaly result from default result index.
          // Here we specify custom AD result index prefix pattern to query all custom result indices.
          resultIndex: _constants.CUSTOM_AD_RESULT_INDEX_PREFIX + '*',
          onlyQueryCustomResultIndex: 'false'
        };
        const aggregationResult = await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: (0, _adHelpers.getResultAggregationQuery)(allDetectorIds, {
            from,
            size,
            sortField,
            sortDirection,
            search,
            indices
          })
        });
        const aggsDetectors = (0, _lodash.get)(aggregationResult, 'aggregations.unique_detectors.buckets', []).reduce((acc, agg) => {
          return {
            ...acc,
            [agg.key]: {
              ...allDetectorsMap[agg.key],
              totalAnomalies: agg.total_anomalies_in_24hr.doc_count,
              lastActiveAnomaly: agg.latest_anomaly_time.value
            }
          };
        }, {});

        // Aggregation will not return values where anomalies for detectors are not generated, loop through it and fill values with 0
        const unUsedDetectors = (0, _lodash.pullAll)(allDetectorIds, Object.keys(aggsDetectors)).reduce((acc, unusedDetector) => {
          return {
            ...acc,
            [unusedDetector]: {
              ...allDetectorsMap[unusedDetector],
              totalAnomalies: 0,
              lastActiveAnomaly: 0
            }
          };
        }, {});

        // If sorting criteria is from the aggregation manage pagination in memory.
        let finalDetectors = (0, _lodash.orderBy)({
          ...aggsDetectors,
          ...unUsedDetectors
        }, [sortField], [sortDirection]);
        if (!sortQueryMap[sortField]) {
          finalDetectors = Object.values(finalDetectors).slice(from, from + size).reduce((acc, detector) => ({
            ...acc,
            [detector.id]: detector
          }), {});
        }

        // Fetch the latest realtime and historical tasks for all detectors
        // using terms aggregations
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};
        try {
          realtimeTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(true)
          });
          historicalTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTasks = (0, _lodash.get)(realtimeTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              realtimeTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});
        const historicalTasks = (0, _lodash.get)(historicalTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              historicalTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});

        // Get real-time and historical task info by looping through each detector & retrieving
        //    - curState by getting real-time task state
        //    - enabledTime by getting real-time task's execution_start time
        //    - taskId by getting historical task's _id
        finalDetectors.forEach(detector => {
          const realtimeTask = (0, _lodash.get)(realtimeTasks[detector.id], 'realtimeTask._source');
          detector.curState = (0, _adHelpers.getTaskState)(realtimeTask);
          detector.enabledTime = (0, _lodash.get)(realtimeTask, 'execution_start_time');
          detector.taskId = (0, _lodash.get)(historicalTasks[detector.id], 'historicalTask._id');
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectorList: Object.values(finalDetectors)
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectorList: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      let {
        id,
        isHistorical,
        resultIndex,
        onlyQueryCustomResultIndex
      } = request.params;
      const {
        dataSourceId = ''
      } = request.params;
      if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
        // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
        resultIndex = '';
      }
      isHistorical = JSON.parse(isHistorical);

      // Search by task id if historical, or by detector id if realtime
      const searchTerm = isHistorical ? {
        task_id: id
      } : {
        detector_id: id
      };
      try {
        const {
          from = 0,
          size = 20,
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = _constants.AD_DOC_FIELDS.DATA_START_TIME,
          startTime = 0,
          endTime = 0,
          fieldName = '',
          anomalyThreshold = -1,
          entityList = ''
        } = request.query;
        const entityListAsObj = entityList.length === 0 ? {} : JSON.parse(entityList);
        const entityFilters = (0, _lodash.isEmpty)(entityListAsObj) ? [] : (0, _adHelpers.getFiltersFromEntityList)(entityListAsObj);

        //Allowed sorting columns
        const sortQueryMap = {
          anomalyGrade: {
            anomaly_grade: sortDirection
          },
          confidence: {
            confidence: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_START_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_START_TIME]: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_END_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_END_TIME]: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }

        //Preparing search request
        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              filter: [{
                term: searchTerm
              }, {
                range: {
                  anomaly_grade: {
                    gt: anomalyThreshold
                  }
                }
              }, ...entityFilters]
            }
          }
        };

        // If querying RT results: remove any results that include a task_id, as this indicates
        // a historical result from a historical task.
        if (!isHistorical) {
          requestBody.query.bool = {
            ...requestBody.query.bool,
            ...{
              must_not: {
                exists: {
                  field: 'task_id'
                }
              }
            }
          };
        }
        try {
          const filterSize = requestBody.query.bool.filter.length;
          if (fieldName) {
            (startTime || endTime) && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.format`, 'epoch_millis');
            startTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.gte`, startTime);
            endTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.lte`, endTime);
          }
        } catch (error) {
          console.log('wrong date range filter', error);
        }
        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('ad.searchResults', {
          body: requestBody
        }) : await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        const totalResults = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectorResult = [];
        const featureResult = {};
        (0, _lodash.get)(response, 'hits.hits', []).forEach(result => {
          detectorResult.push({
            startTime: result._source.data_start_time,
            endTime: result._source.data_end_time,
            plotTime: result._source.data_end_time,
            contributions: result._source.anomaly_grade > 0 ? result._source.relevant_attribution : {},
            confidence: result._source.confidence != null && result._source.confidence !== 'NaN' && result._source.confidence > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.confidence)) : 0,
            anomalyGrade: result._source.anomaly_grade != null && result._source.anomaly_grade !== 'NaN' && result._source.anomaly_grade > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.anomaly_grade)) : 0,
            ...(result._source.entity != null ? {
              entity: result._source.entity
            } : {}),
            // TODO: we should refactor other places to read feature data from
            // AnomalyResult, instead of having separate FeatureData which is hard
            // to know feature data belongs to which anomaly result
            features: this.getFeatureData(result)
          });
          result._source.feature_data.forEach(featureData => {
            if (!featureResult[featureData.feature_id]) {
              featureResult[featureData.feature_id] = [];
            }
            featureResult[featureData.feature_id].push({
              startTime: result._source.data_start_time,
              endTime: result._source.data_end_time,
              plotTime: result._source.data_end_time,
              data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
              name: featureData.feature_name,
              expectedValue: this.getExpectedValue(result, featureData)
            });
          });
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalAnomalies: totalResults,
              results: detectorResult,
              featureResults: featureResult
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get results', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getTopAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.topHistoricalAnomalyResults' : 'ad.topAnomalyResults';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          detectorId: detectorId,
          body: request.body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getTopAnomalyResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "matchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorName
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.matchDetector', {
          detectorName
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - matchDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectorCount", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.detectorCount');
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getDetectorCount', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getFeatureData", rawResult => {
      const featureResult = {};
      rawResult._source.feature_data.forEach(featureData => {
        featureResult[featureData.feature_id] = {
          startTime: rawResult._source.data_start_time,
          endTime: rawResult._source.data_end_time,
          plotTime: rawResult._source.data_end_time,
          data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
          name: featureData.feature_name,
          expectedValue: this.getExpectedValue(rawResult, featureData)
        };
      });
      return featureResult;
    });
    _defineProperty(this, "getExpectedValue", (rawResult, featureData) => {
      let expectedValue = featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0;
      if (rawResult._source.anomaly_grade > 0) {
        const expectedValueList = rawResult._source.expected_values;
        if ((expectedValueList === null || expectedValueList === void 0 ? void 0 : expectedValueList.length) > 0) {
          expectedValueList[0].value_list.forEach(expect => {
            if (expect.feature_id === featureData.feature_id) {
              expectedValue = expect.data;
            }
          });
        }
      }
      return expectedValue;
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = AdService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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