"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configRoutes = configRoutes;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _helper = require("../utils/helper");
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../common/constants");
var _MDSEnabledClientService = require("../../common/MDSEnabledClientService");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function configRoutes(router, dataSourceEnabled) {
  const getConfigsQuerySchema = {
    from_index: _configSchema.schema.number(),
    max_items: _configSchema.schema.number(),
    query: _configSchema.schema.maybe(_configSchema.schema.string()),
    config_type: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()]),
    is_enabled: _configSchema.schema.maybe(_configSchema.schema.boolean()),
    sort_field: _configSchema.schema.string(),
    sort_order: _configSchema.schema.string(),
    config_id_list: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])),
    'smtp_account.method': _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()]))
  };
  if (dataSourceEnabled) {
    getConfigsQuerySchema.dataSourceId = _configSchema.schema.string();
  }
  const genericBodyAndDataSourceIdQuery = {
    body: _configSchema.schema.any()
  };
  if (dataSourceEnabled) {
    genericBodyAndDataSourceIdQuery.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const genericParamsAndDataSourceIdQuery = {
    params: _configSchema.schema.any()
  };
  if (dataSourceEnabled) {
    genericParamsAndDataSourceIdQuery.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const updateQuerySchema = {
    body: _configSchema.schema.any(),
    params: _configSchema.schema.object({
      configId: _configSchema.schema.string()
    })
  };
  if (dataSourceEnabled) {
    updateQuerySchema.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const deleteQuerySchema = {
    config_id_list: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])
  };
  if (dataSourceEnabled) {
    deleteQuerySchema.dataSourceId = _configSchema.schema.string();
  }
  router.get({
    path: _common.NODE_API.GET_CONFIGS,
    validate: {
      query: _configSchema.schema.object(getConfigsQuerySchema)
    }
  }, async (context, request, response) => {
    const config_type = (0, _helper.joinRequestParams)(request.query.config_type);
    const config_id_list = (0, _helper.joinRequestParams)(request.query.config_id_list);
    const encryption_method = (0, _helper.joinRequestParams)(request.query['smtp_account.method']);
    const query = request.query.query;
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getConfigs', {
        from_index: request.query.from_index,
        max_items: request.query.max_items,
        is_enabled: request.query.is_enabled,
        sort_field: request.query.sort_field,
        sort_order: request.query.sort_order,
        config_type,
        ...(query && {
          text_query: query
        }),
        // text_query will exclude keyword fields
        ...(config_id_list && {
          config_id_list
        }),
        ...(encryption_method && {
          'smtp_account.method': encryption_method
        })
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 404,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_common.NODE_API.GET_CONFIG}/{configId}`,
    validate: genericParamsAndDataSourceIdQuery
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getConfigById', {
        configId: request.params.configId
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _common.NODE_API.CREATE_CONFIG,
    validate: genericBodyAndDataSourceIdQuery
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.createConfig', {
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_common.NODE_API.UPDATE_CONFIG}/{configId}`,
    validate: updateQuerySchema
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.updateConfigById', {
        configId: request.params.configId,
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: _common.NODE_API.DELETE_CONFIGS,
    validate: {
      query: _configSchema.schema.object(deleteQuerySchema)
    }
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    const config_id_list = (0, _helper.joinRequestParams)(request.query.config_id_list);
    try {
      const resp = await client('notifications.deleteConfigs', {
        config_id_list
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: _common.NODE_API.GET_AVAILABLE_FEATURES,
    validate: dataSourceEnabled ? {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.string()
      })
    } : false
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getServerFeatures');
      const config_type_list = resp.allowed_config_type_list;
      const channelTypes = {};
      for (let channel of config_type_list) {
        if (_constants.CHANNEL_TYPE[channel]) {
          channelTypes[channel] = _constants.CHANNEL_TYPE[channel];
        }
      }
      const availableFeature = {
        availableChannels: channelTypes,
        availableConfigTypes: config_type_list,
        tooltipSupport: _lodash.default.get(response, ['plugin_features', 'tooltip_support']) === 'true'
      };
      return response.ok({
        body: availableFeature
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 404,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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