"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deepCheck = deepCheck;
exports.pruneConfig = exports.foldResults = void 0;
exports.tryParseNDJson = tryParseNDJson;
var _stream = require("stream");
var _utils = require("../../../../../../src/core/server/saved_objects/routes/utils");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

async function tryParseNDJson(content) {
  try {
    const objects = await (0, _utils.createSavedObjectsStreamFromNdJson)(_stream.Readable.from(content));
    return await objects.toArray();
  } catch (err) {
    return null;
  }
}

/**
 * Check IntegrationReader nested dependencies for validity,
 * as a supplement to shallow config validation.
 *
 * @returns a Result indicating whether the integration is valid, holding the integration's config.
 */
async function deepCheck(reader) {
  const configResult = await reader.getConfig();
  if (!configResult.ok) {
    return configResult;
  }

  // Deep checks not included in default config validation
  const assets = await reader.getAssets();
  if (!assets.ok || assets.value.length === 0) {
    return {
      ok: false,
      error: new Error('An integration must have at least one asset')
    };
  }
  return configResult;
}

/**
 * Helper method: Convert an Array<Result<type>> to Result<Array<type>>.
 *
 * @param results The list of results to fold.
 * @returns A single result object with values in an array, or an error result.
 */
const foldResults = results => results.reduce((result, currentValue) => {
  if (!result.ok) {
    return result;
  }
  if (!currentValue.ok) {
    return currentValue;
  }
  result.value.push(currentValue.value);
  return result;
}, {
  ok: true,
  value: []
});

/**
 * Remove all fields from SerializedIntegration not present in IntegrationConfig.
 *
 * @param rawConfig The raw config to prune
 * @returns A config with all data fields removed
 */
exports.foldResults = foldResults;
const pruneConfig = rawConfig => {
  // Hacky workaround: we currently only need to prune 'data' fields, so just remove every 'data'.
  // Lots of risky conversion in this method, so scope it to here and rewrite if more granular
  // pruning is needed.
  const prunePart = part => {
    const result = {};
    for (const [key, value] of Object.entries(part)) {
      if (key === 'data') {
        continue;
      } else if (Array.isArray(value)) {
        result[key] = value.map(item => {
          if (item instanceof Object && item !== null) {
            return prunePart(item);
          }
          return item;
        });
      } else if (value instanceof Object && value !== null) {
        result[key] = prunePart(value);
      } else {
        result[key] = value;
      }
    }
    return result;
  };
  return prunePart(rawConfig);
};
exports.pruneConfig = pruneConfig;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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