"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationReader = void 0;
var _path = _interopRequireDefault(require("path"));
var _semver = _interopRequireDefault(require("semver"));
var _validators = require("../validators");
var _fs_data_adaptor = require("./fs_data_adaptor");
var _utils = require("./utils");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * The Integration class represents the data for Integration Templates.
 * It is backed by the repository file system.
 * It includes accessor methods for integration configs, as well as helpers for nested components.
 */
class IntegrationReader {
  constructor(directory, reader) {
    _defineProperty(this, "reader", void 0);
    _defineProperty(this, "directory", void 0);
    _defineProperty(this, "name", void 0);
    this.directory = directory;
    this.name = _path.default.basename(directory);
    this.reader = reader !== null && reader !== void 0 ? reader : new _fs_data_adaptor.FileSystemDataAdaptor(directory);
  }

  /**
   * Retrieve data from correct source regardless of if reader is config-localized or not.
   *
   * TODO refactor to assemble filename from `type` instead of requiring caller to format it.
   *
   * @param item An item which may have data in it.
   * @param fileParams Information about the file to read if the config is not localized.
   * @param format How to package the returned data.
   *               If 'json', return `object | object[]`. If 'binary', return `Buffer`.
   * @returns A result with the data, with a format based on the format field.
   */

  async fetchDataOrReadFile(item, fileParams, format) {
    if (this.reader.isConfigLocalized) {
      if (!item.data) {
        return {
          ok: false,
          error: new Error('The config for the provided reader is localized, but no data field is present. ' + JSON.stringify(item))
        };
      }
      try {
        if (format === 'json') {
          return {
            ok: true,
            value: JSON.parse(item.data)
          };
        } else {
          return {
            ok: true,
            value: Buffer.from(item.data, 'base64')
          };
        }
      } catch (error) {
        return {
          ok: false,
          error
        };
      }
    }
    if (format === 'json') {
      return this.reader.readFile(fileParams.filename, fileParams.type);
    } else {
      return this.reader.readFileRaw(fileParams.filename, fileParams.type);
    }
  }
  async readAsset(asset) {
    const filename = `${asset.name}-${asset.version}.${asset.extension}`;
    const fileParams = {
      filename,
      type: 'assets'
    };
    if (['json', 'ndjson'].includes(asset.extension)) {
      const maybeObject = await this.fetchDataOrReadFile(asset, fileParams, 'json');
      if (!maybeObject.ok) {
        return maybeObject;
      }
      return {
        ok: true,
        value: {
          ...asset,
          data: JSON.stringify(maybeObject.value)
        }
      };
    } else {
      const maybeBuffer = await this.fetchDataOrReadFile(asset, fileParams, 'binary');
      if (!maybeBuffer.ok) {
        return maybeBuffer;
      }
      return {
        ok: true,
        value: {
          ...asset,
          data: maybeBuffer.value.toString('utf8')
        }
      };
    }
  }

  /**
   * Get the latest version of the integration available.
   * This method relies on the fact that integration configs have their versions in their name.
   * Any files that don't match the config naming convention will be ignored.
   *
   * @returns A string with the latest version, or null if no versions are available.
   */
  async getLatestVersion() {
    const versions = await this.reader.findIntegrationVersions();
    if (!versions.ok) {
      return null;
    }
    if (versions.value.length === 0) {
      return null;
    }
    // Sort descending
    versions.value.sort(_semver.default.rcompare);
    return versions.value[0];
  }

  // Get config without pruning or validation.
  async getRawConfig(version) {
    if ((await this.reader.getDirectoryType()) !== 'integration') {
      return {
        ok: false,
        error: new Error(`${this.directory} is not a valid integration directory`)
      };
    }
    const maybeVersion = version ? version : await this.getLatestVersion();
    if (maybeVersion === null) {
      return {
        ok: false,
        error: new Error(`No valid config matching version ${version} is available`)
      };
    }
    const configFile = `${this.name}-${maybeVersion}.json`;

    // Even config-localized readers must support config-read.
    const config = await this.reader.readFile(configFile);
    if (!config.ok) {
      return config;
    }
    return (0, _validators.validateTemplate)(config.value);
  }

  /**
   * Get the configuration of the current integration.
   *
   * @param version The version of the config to retrieve.
   * @returns The config if a valid config matching the version is present, otherwise null.
   */
  async getConfig(version) {
    const maybeConfig = await this.getRawConfig(version);
    if (!maybeConfig.ok) {
      return maybeConfig;
    }
    return (0, _validators.validateTemplate)((0, _utils.pruneConfig)(maybeConfig.value));
  }

  /**
   * Retrieve assets associated with the integration.
   * This method greedily retrieves all assets.
   * If an asset is invalid, an error result is returned.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns A result containing the parsed assets.
   */
  async getAssets(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }
    const config = configResult.value;
    const resultValue = [];
    for (const asset of config.assets) {
      const serializedResult = await this.readAsset(asset);
      if (!serializedResult.ok) {
        return serializedResult;
      }
      switch (asset.type) {
        case 'savedObjectBundle':
          resultValue.push({
            type: 'savedObjectBundle',
            workflows: asset.workflows,
            data: JSON.parse(serializedResult.value.data)
          });
          break;
        case 'query':
          resultValue.push({
            type: 'query',
            workflows: asset.workflows,
            query: serializedResult.value.data,
            language: asset.extension
          });
          break;
      }
    }
    return {
      ok: true,
      value: resultValue
    };
  }

  /**
   * Retrieve sample data associated with the integration.
   * If the version is invalid, an error is thrown.
   * If the sample data is invalid, null will be returned
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing a list of sample data with adjusted timestamps.
   */
  async getSampleData(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }
    const config = configResult.value;
    const resultValue = {
      sampleData: null
    };
    if (config.sampleData) {
      const jsonContent = await this.fetchDataOrReadFile(config.sampleData, {
        filename: config.sampleData.path,
        type: 'data'
      }, 'json');
      if (!jsonContent.ok) {
        return jsonContent;
      }
      for (const value of jsonContent.value) {
        if (!('@timestamp' in value)) {
          continue;
        }
        // Randomly scatter timestamps across last 10 minutes
        // Assume for now that the ordering of events isn't important, can change to a sequence if needed
        // Also doesn't handle fields like `observedTimestamp` if present
        const newTime = new Date(Date.now() - Math.floor(Math.random() * 1000 * 60 * 10)).toISOString();
        Object.assign(value, {
          '@timestamp': newTime
        });
        if ('observedTimestamp' in value) {
          Object.assign(value, {
            observedTimestamp: newTime
          });
        }
      }
      resultValue.sampleData = jsonContent.value;
    }
    return {
      ok: true,
      value: resultValue
    };
  }

  /**
   * Retrieve schema data associated with the integration.
   * This method greedily retrieves all mappings and schemas.
   * It's assumed that a valid version will be provided.
   * If the version is invalid, an error is thrown.
   * If a schema is invalid, an error will be thrown.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing the different types of assets.
   */
  async getSchemas(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }
    const config = configResult.value;
    const resultValue = {
      mappings: {}
    };
    for (const component of config.components) {
      const schemaFile = `${component.name}-${component.version}.mapping.json`;
      const schema = await this.fetchDataOrReadFile(component, {
        filename: schemaFile,
        type: 'schemas'
      }, 'json');
      if (!schema.ok) {
        return schema;
      }
      resultValue.mappings[component.name] = schema.value;
    }
    return {
      ok: true,
      value: resultValue
    };
  }

  /**
   * Retrieves the data for a static file associated with the integration.
   *
   * @param staticPath The path of the static to retrieve.
   * @returns A buffer with the static's data if present, otherwise null.
   */
  async getStatic(staticPath) {
    // Statics were originally designed to read straight from file system,
    // so we use direct access if possible.
    if (!this.reader.isConfigLocalized) {
      return await this.reader.readFileRaw(staticPath, 'static');
    }

    // Otherwise, we need to search for the right static, by checking each version.
    const versions = await this.reader.findIntegrationVersions();
    if (!versions.ok) {
      return versions;
    }
    for (const version of versions.value) {
      var _statics$logo, _statics$darkModeLogo;
      const config = await this.getRawConfig(version);
      if (!config.ok || !config.value.statics) {
        continue;
      }
      const statics = config.value.statics;
      if (((_statics$logo = statics.logo) === null || _statics$logo === void 0 ? void 0 : _statics$logo.path) === staticPath) {
        if (!('data' in statics.logo)) {
          return {
            ok: false,
            error: new Error('Localized config missing static data')
          };
        }
        return {
          ok: true,
          value: Buffer.from(statics.logo.data, 'base64')
        };
      }
      if ((statics === null || statics === void 0 || (_statics$darkModeLogo = statics.darkModeLogo) === null || _statics$darkModeLogo === void 0 ? void 0 : _statics$darkModeLogo.path) === staticPath) {
        if (!('data' in statics.darkModeLogo)) {
          return {
            ok: false,
            error: new Error('Localized config missing static data')
          };
        }
        return {
          ok: true,
          value: Buffer.from(statics.darkModeLogo.data, 'base64')
        };
      }
      for (const iterStatic of [...((_statics$gallery = statics === null || statics === void 0 ? void 0 : statics.gallery) !== null && _statics$gallery !== void 0 ? _statics$gallery : []), ...((_statics$darkModeGall = statics === null || statics === void 0 ? void 0 : statics.darkModeGallery) !== null && _statics$darkModeGall !== void 0 ? _statics$darkModeGall : [])]) {
        var _statics$gallery, _statics$darkModeGall;
        if (iterStatic.path === staticPath) {
          if (!('data' in iterStatic)) {
            return {
              ok: false,
              error: new Error('Localized config missing static data')
            };
          }
          return {
            ok: true,
            value: Buffer.from(iterStatic.data, 'base64')
          };
        }
      }
    }
    return {
      ok: false,
      error: new Error(`Static not found: ${staticPath}`, {
        code: 'ENOENT'
      })
    };
  }
  async serializeStaticAsset(asset) {
    const data = await this.getStatic(asset.path);
    if (!data.ok) {
      return data;
    }
    return {
      ok: true,
      value: {
        ...asset,
        data: data.value.toString('base64')
      }
    };
  }
  async serializeStatics(statics) {
    const serialized = {};
    if (statics.logo) {
      const serializeResult = await this.serializeStaticAsset(statics.logo);
      serialized.logo = serializeResult.value;
    }
    if (statics.darkModeLogo) {
      const serializeResult = await this.serializeStaticAsset(statics.darkModeLogo);
      serialized.darkModeLogo = serializeResult.value;
    }
    if (statics.gallery) {
      const results = await Promise.all(statics.gallery.map(asset => this.serializeStaticAsset(asset)));
      const foldedResult = (0, _utils.foldResults)(results);
      serialized.gallery = foldedResult.value;
    }
    if (statics.darkModeGallery) {
      const results = await Promise.all(statics.darkModeGallery.map(asset => this.serializeStaticAsset(asset)));
      const foldedResult = (0, _utils.foldResults)(results);
      serialized.darkModeGallery = foldedResult.value;
    }
    return {
      ok: true,
      value: serialized
    };
  }

  /**
   * Serialize the referenced integration as a flat JSON object.
   * Useful for normalizing the format for sending to other locations.
   * This method implements the serialization scheme expected by `JsonCatalogDataAdaptor`.
   *
   * @param version The version of the integration to serialize.
   * @returns A large object which includes all of the integration's data.
   */
  async serialize(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }

    // Type cast safety: all serializable properties must have the 'data' field.
    // The remainder of the method is populating all such fields.
    const config = configResult.value;
    const componentResults = await Promise.all(config.components.map(component => this.fetchDataOrReadFile(component, {
      filename: `${component.name}-${component.version}.mapping.json`,
      type: 'schemas'
    }, 'json')));
    const componentsResult = (0, _utils.foldResults)(componentResults);
    if (!componentsResult.ok) {
      return componentsResult;
    }
    config.components = config.components.map((component, idx) => {
      return {
        ...component,
        data: JSON.stringify(componentsResult.value[idx])
      };
    });
    const assetResults = await Promise.all(config.assets.map(asset => this.readAsset(asset)));
    const assets = (0, _utils.foldResults)(assetResults);
    if (!assets.ok) {
      return assets;
    }
    config.assets = assets.value;
    if (config.statics) {
      const staticsResult = await this.serializeStatics(config.statics);
      if (!staticsResult.ok) {
        return staticsResult;
      }
      config.statics = staticsResult.value;
    }
    if (config.sampleData) {
      const dataResult = await this.getSampleData(version);
      if (!dataResult.ok) {
        return dataResult;
      }
      config.sampleData = {
        ...config.sampleData,
        data: JSON.stringify(dataResult.value.sampleData)
      };
    }
    return {
      ok: true,
      value: config
    };
  }
}
exports.IntegrationReader = IntegrationReader;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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